<?php
/**
 * Statistics Tracking Class.
 *
 * Tracks and manages WebP conversion statistics.
 *
 * @package WP_WebP_Optimizer
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Statistics Class.
 */
class WP_WebP_Optimizer_Stats {

	/**
	 * Get all statistics.
	 *
	 * @since 1.0.0
	 * @return array Statistics array.
	 */
	public static function get_stats() {
		$stats = get_option( 'wp_webp_stats', array(
			'total_conversions'      => 0,
			'total_space_saved'      => 0,
			'successful_conversions' => 0,
			'failed_conversions'     => 0,
			'average_conversion_time' => 0,
			'last_conversion_date'   => null,
		) );

		// Get real-time counts from database
		$stats['total_images'] = self::get_total_images();
		$stats['converted_images'] = self::get_converted_count();
		$stats['pending_images'] = $stats['total_images'] - $stats['converted_images'];
		$stats['average_compression'] = self::get_average_compression();

		return $stats;
	}

	/**
	 * Get total image count.
	 *
	 * @since 1.0.0
	 * @return int Total count.
	 */
	public static function get_total_images() {
		$args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ),
			'post_status'    => 'inherit',
			'posts_per_page' => -1,
			'fields'         => 'ids',
		);

		$query = new WP_Query( $args );
		return count( $query->posts );
	}

	/**
	 * Get converted images count.
	 *
	 * @since 1.0.0
	 * @return int Converted count.
	 */
	public static function get_converted_count() {
		$args = array(
			'post_type'      => 'attachment',
			'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ),
			'post_status'    => 'inherit',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'meta_query'     => array(
				array(
					'key'     => '_webp_converted',
					'value'   => '1',
					'compare' => '=',
				),
			),
		);

		$query = new WP_Query( $args );
		return count( $query->posts );
	}

	/**
	 * Get average compression ratio.
	 *
	 * @since 1.0.0
	 * @return float Average compression percentage.
	 */
	public static function get_average_compression() {
		global $wpdb;

		$results = $wpdb->get_results(
			"SELECT meta_value as original_size FROM {$wpdb->postmeta} WHERE meta_key = '_webp_original_size'",
			ARRAY_A
		);

		$results_new = $wpdb->get_results(
			"SELECT meta_value as new_size FROM {$wpdb->postmeta} WHERE meta_key = '_webp_new_size'",
			ARRAY_A
		);

		if ( empty( $results ) || empty( $results_new ) ) {
			return 0;
		}

		$total_original = array_sum( array_column( $results, 'original_size' ) );
		$total_new = array_sum( array_column( $results_new, 'new_size' ) );

		if ( $total_original == 0 ) {
			return 0;
		}

		return round( ( 1 - ( $total_new / $total_original ) ) * 100, 2 );
	}

	/**
	 * Get recent conversion activity.
	 *
	 * @since 1.0.0
	 * @param int $limit Number of entries to retrieve.
	 * @return array Recent conversions.
	 */
	public static function get_recent_conversions( $limit = 10 ) {
		global $wpdb;

		$results = $wpdb->get_results(
			$wpdb->prepare(
				"SELECT p.ID, p.post_title, pm1.meta_value as conversion_date, pm2.meta_value as original_size, pm3.meta_value as new_size
				FROM {$wpdb->posts} p
				INNER JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_webp_conversion_date'
				LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_webp_original_size'
				LEFT JOIN {$wpdb->postmeta} pm3 ON p.ID = pm3.post_id AND pm3.meta_key = '_webp_new_size'
				WHERE p.post_type = 'attachment'
				ORDER BY pm1.meta_value DESC
				LIMIT %d",
				$limit
			),
			ARRAY_A
		);

		return $results;
	}

	/**
	 * Get conversion logs.
	 *
	 * @since 1.0.0
	 * @param int $page     Page number.
	 * @param int $per_page Items per page.
	 * @return array Logs array with pagination data.
	 */
	public static function get_conversion_logs( $page = 1, $per_page = 50 ) {
		$logs = get_option( 'wp_webp_conversion_logs', array() );
		$total = count( $logs );

		// Reverse to show newest first
		$logs = array_reverse( $logs );

		// Paginate
		$offset = ( $page - 1 ) * $per_page;
		$paged_logs = array_slice( $logs, $offset, $per_page );

		return array(
			'logs'        => $paged_logs,
			'total'       => $total,
			'total_pages' => ceil( $total / $per_page ),
			'current_page' => $page,
		);
	}

	/**
	 * Clear all conversion logs.
	 *
	 * @since 1.0.0
	 * @return bool Success status.
	 */
	public static function clear_logs() {
		delete_option( 'wp_webp_conversion_logs' );
		return true;
	}

	/**
	 * Export logs to CSV.
	 *
	 * @since 1.0.0
	 * @return string CSV content.
	 */
	public static function export_logs_csv() {
		$logs = get_option( 'wp_webp_conversion_logs', array() );

		$csv = "Date,Image ID,Status,Original Size (bytes),New Size (bytes),Savings (bytes),Message\n";

		foreach ( $logs as $log ) {
			$savings = isset( $log['original_size'], $log['new_size'] ) 
				? $log['original_size'] - $log['new_size'] 
				: 0;

			$csv .= sprintf(
				"%s,%d,%s,%d,%d,%d,%s\n",
				isset( $log['time'] ) ? $log['time'] : '',
				isset( $log['id'] ) ? $log['id'] : 0,
				isset( $log['status'] ) ? $log['status'] : '',
				isset( $log['original_size'] ) ? $log['original_size'] : 0,
				isset( $log['new_size'] ) ? $log['new_size'] : 0,
				$savings,
				isset( $log['message'] ) ? str_replace( ',', ';', $log['message'] ) : ''
			);
		}

		return $csv;
	}

	/**
	 * Get conversion errors.
	 *
	 * @since 1.0.0
	 * @return array Error logs.
	 */
	public static function get_errors() {
		return get_option( 'wp_webp_conversion_errors', array() );
	}

	/**
	 * Get total space saved in human-readable format.
	 *
	 * @since 1.0.0
	 * @return string Formatted size.
	 */
	public static function get_formatted_space_saved() {
		$stats = self::get_stats();
		return size_format( $stats['total_space_saved'], 2 );
	}

	/**
	 * Reset all statistics.
	 *
	 * @since 1.0.0
	 * @return bool Success status.
	 */
	public static function reset_stats() {
		$default_stats = array(
			'total_conversions'      => 0,
			'total_space_saved'      => 0,
			'successful_conversions' => 0,
			'failed_conversions'     => 0,
			'average_conversion_time' => 0,
			'last_conversion_date'   => null,
		);

		update_option( 'wp_webp_stats', $default_stats );
		delete_option( 'wp_webp_conversion_logs' );
		delete_option( 'wp_webp_conversion_errors' );

		return true;
	}
}


